// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef COREBASE_H
#define COREBASE_H

#include "interrupt.h"
#include "hardware.h"
#include "oparray.h"
#include "registers.h"
#include "emutime.h"
#include <hash_map>

//Illegal Primary/Secondary Opcodes used by WhineCube.
//I assume that they're uncommon enough in real programs that
//the risk of them fouling up the system is negligible.
#define IPO_DYNAREC1 1
#define IPO_DYNAREC2 6
#define IPO_OOVPA 5
#define IPO_NOARG 2
#define ISO_WC_DEGUB 0
#define ISO_GCM_DOL_LOADED 1
#define ISO_OSREPORT 2
#define ISO_DSP_QUICK_EXECUTE 3

#define OPCODES(macro) \
	macro(stw, 36, 0, 0)\
	macro(andi, 28, 0, 0)\
	macro(addi, 14, 0, 0)\
	macro(stwu, 37, 0, 0)\
	macro(b, 18, 0, 0)\
	macro(addis, 15, 0, 0)\
	macro(ori, 24, 0, 0)\
	macro(bclr, 19, 21, 16)\
	macro(dcbf, 31, 21, 86)\
	macro(mfmsr, 31, 21, 83)\
	macro(mtmsr, 31, 21, 146)\
	macro(mfspr, 31, 21, 339)\
	macro(oris, 25, 0, 0)\
	macro(or, 31, 21, 444)\
	macro(mtspr, 31, 21, 467)\
	macro(bc, 16, 0, 0)\
	macro(lwz, 32, 0, 0)\
	macro(sync, 31, 21, 598)\
	macro(rlwinm, 21, 0, 0)\
	macro(cmpli, 10, 0, 0)\
	macro(isync, 19, 21, 150)\
	macro(crxor, 19, 21, 193)\
	macro(add, 31, 22, 266)\
	macro(subf, 31, 22, 40)\
	macro(cmp, 31, 21, 0)\
	macro(xori, 26, 0, 0)\
	macro(cmpi, 11, 0, 0)\
	macro(lhz, 40, 0, 0)\
	macro(stb, 38, 0, 0)\
	macro(lbz, 34, 0, 0)\
	macro(mulli, 7, 0, 0)\
	macro(mulhw, 31, 22, 75)\
	macro(srawi, 31, 21, 824)\
	macro(stfd, 54, 0, 0)\
	macro(lfd, 50, 0, 0)\
	macro(extsb, 31, 21, 954)\
	macro(mtfsb1, 63, 21, 38)\
	macro(andc, 31, 21, 60)\
	macro(cntlzw, 31, 21, 26)\
	macro(sth, 44, 0, 0)\
	macro(and, 31, 21, 28)\
	macro(xor, 31, 21, 316)\
	macro(neg, 31, 22, 104)\
	macro(extsh, 31, 21, 922)\
	macro(addze, 31, 22, 202)\
	macro(mtfsf, 63, 21, 711)\
	macro(lwzx, 31, 21, 23)\
	macro(subfic, 8, 0, 0)\
	macro(lfs, 48, 0, 0)\
	macro(stwx, 31, 21, 151)\
	macro(xoris, 27, 0, 0)\
	macro(fsub, 63, 26, 20)\
	macro(addic, 12, 0, 0)\
	macro(subfe, 31, 22, 136)\
	macro(lbzx, 31, 21, 87)\
	macro(mullw, 31, 22, 235)\
	macro(lha, 42, 0, 0)\
	macro(stbx, 31, 21, 215)\
	macro(sthx, 31, 21, 407)\
	macro(bcctr, 19, 21, 528)\
	macro(slw, 31, 21, 24)\
	macro(sraw, 31, 21, 792)\
	macro(divw, 31, 22, 491)\
	macro(adde, 31, 22, 138)\
	macro(cmpl, 31, 21, 32)\
	macro(mcrf, 19, 21, 0)\
	macro(mfcr, 31, 21, 19)\
	macro(mtcrf, 31, 21, 144)\
	macro(stfs, 52, 0, 0)\
	macro(fadds, 59, 26, 21)\
	macro(fsubs, 59, 26, 20)\
	macro(cror, 19, 21, 449)\
	macro(subfc, 31, 22, 8)\
	macro(lbzu, 35, 0, 0)\
	macro(nor, 31, 21, 124)\
	macro(lwzu, 33, 0, 0)\
	macro(dcbst, 31, 21, 54)\
	macro(sc, 17, 0, 0)\
	macro(mftb, 31, 21, 371)\
	macro(addc, 31, 22, 10)\
	macro(stmw, 47, 0, 0)\
	macro(icbi, 31, 21, 982)\
	macro(lmw, 46, 0, 0)\
	macro(lhzx, 31, 21, 279)\
	macro(mulhwu, 31, 22, 11)\
	macro(rlwimi, 20, 0, 0)\
	macro(dcbi, 31, 21, 470)\
	macro(stbu, 39, 0, 0)\
	macro(nand, 31, 21, 476)\
	macro(rfi, 19, 21, 50)\
	macro(divwu, 31, 22, 459)\
	macro(srw, 31, 21, 536)\
	macro(stfsx, 31, 21, 663)\
	macro(lhax, 31, 21, 343)\
	macro(lfsx, 31, 21, 535)\
	macro(fneg, 63, 21, 40)\
	macro(fmul, 63, 26, 25)\
	macro(frsp, 63, 21, 12)\
	macro(fdivs, 59, 26, 18)\
	macro(fmsubs, 59, 26, 28)\
	macro(fmadds, 59, 26, 29)\
	macro(fmuls, 59, 26, 25)\
	macro(fmr, 63, 21, 72)\
	macro(fcmpu, 63, 21, 0)\
	macro(fctiwz, 63, 21, 15)\
	macro(fadd, 63, 26, 21)\
	macro(fmadd, 63, 26, 29)\
	macro(lfdx, 31, 21, 599)\
	macro(addme, 31, 22, 234)\
	macro(crnor, 19, 21, 33)\
	macro(fdiv, 63, 26, 18)\
	macro(andis, 29, 0, 0)\
	macro(twi, 3, 0, 0)\
	macro(fmsub, 63, 26, 28)\
	macro(subfze, 31, 22, 200)\
	macro(orc, 31, 21, 412)\
	macro(lhzu, 41, 0, 0)\
	macro(lfdu, 51, 0, 0)\
	macro(dcbz, 31, 21, 1014)\
	macro(lbzux, 31, 21, 119)\
	macro(psq_lu, 57, 0, 0)\
	macro(sthu, 45, 0, 0)\
	macro(fnmadd, 63, 26, 31)\
	macro(creqv, 19, 21, 289)\
	macro(fabs, 63, 21, 264)\
	macro(mtsrin, 31, 21, 242)\
	macro(mfsr, 31, 21, 595)\
	macro(mffs, 63, 21, 583)\
	macro(tlbie, 31, 21, 306)\
	macro(stswi, 31, 21, 725)\
	macro(lwarx, 31, 21, 20)\
	macro(lwzux, 31, 21, 55)\
	macro(dcbt, 31, 21, 278)\
	macro(lswi, 31, 21, 597)\
	macro(dcbtst, 31, 21, 246)\
	macro(rlwnm, 23, 0, 0)\
	macro(stwux, 31, 21, 183)\
	macro(stwbrx, 31, 21, 662)\
	macro(mfsrin, 31, 21, 659)\
	macro(fsel, 63, 26, 23)\
	macro(psq_l, 56, 0, 0)\
	macro(psq_st, 60, 0, 0)\
	macro(fcmpo, 63, 21, 32)\
	macro(ps_merge01, 4, 21, 560)\
	macro(ps_merge10, 4, 21, 592)\
	macro(ps_mr, 4, 21, 72)\
	macro(stfsu, 53, 0, 0)\
	macro(stfdx, 31, 21, 727)\
	macro(stbux, 31, 21, 247)\
	macro(ps_mul, 4, 26, 25)\
	macro(ps_madd, 4, 26, 29)\
	macro(ps_sum0, 4, 26, 10)\
	macro(frsqrte, 63, 26, 26)\
	macro(fnmsubs, 59, 26, 30)\
	macro(ps_muls0, 4, 26, 12)\
	macro(ps_muls1, 4, 26, 13)\
	macro(ps_msub, 4, 26, 28)\
	macro(ps_neg, 4, 21, 40)\
	macro(lfsu, 49, 0, 0)\
	macro(eieio, 31, 21, 854)\
	macro(ps_merge00, 4, 21, 528)\
	macro(ps_merge11, 4, 21, 624)\
	macro(fnmadds, 59, 26, 31)\
	macro(ps_madds0, 4, 26, 14)\
	macro(ps_madds1, 4, 26, 15)\
	macro(ps_add, 4, 26, 21)\
	macro(ps_sub, 4, 26, 20)\
	macro(ps_cmpo0, 4, 21, 32)\
	macro(ps_cmpo1, 4, 21, 96)\
	macro(ps_cmpu0, 4, 21, 0)\
	macro(ps_cmpu1, 4, 21, 64)\
	macro(fres, 59, 26, 24)\
	macro(ps_nmadd, 4, 26, 31)\
	macro(ps_nmsub, 4, 26, 30)\
	macro(ps_sum1, 4, 26, 11)\
	macro(eqv, 31, 21, 284)\
	macro(lhau, 43, 0, 0)\
	macro(lhaux, 31, 21, 375)\
	macro(fnmsub, 63, 26, 30)\
	macro(psq_stu, 61, 0, 0)\
	macro(ps_rsqrte, 4, 26, 26)\
	macro(ps_res, 4, 26, 24)\
	macro(mcrxr, 31, 21, 512)\
	macro(stfiwx, 31, 21, 983)\
	macro(fctiw, 63, 21, 14)\
	macro(sthbrx, 31, 21, 918)\
	macro(stfdu, 55, 0, 0)\
	macro(stfdux, 31, 21, 759)\
	macro(mtfsb0, 63, 21, 70)\
	macro(dcbz_l, 4, 21, 1014)\
	macro(lwbrx, 31, 21, 534)\
	macro(lhbrx, 31, 21, 790)\
	macro(lfdux, 31, 21, 631)\
	macro(lfsux, 31, 21, 567)\
	macro(lhzux, 31, 21, 311)\
	macro(psq_lux, 4, 25, 38)\
	macro(psq_lx, 4, 25, 6)\
	macro(psq_stux, 4, 25, 39)\
	macro(psq_stx, 4, 25, 7)\
	macro(mtsr, 31, 21, 210)\
	macro(ps_abs, 4, 21, 264)\
	macro(ps_div, 4, 26, 18)\
	macro(ps_nabs, 4, 21, 136)\
	macro(ps_sel, 4, 26, 23)\
	macro(tlbsync, 31, 21, 566)\
	macro(subfme, 31, 22, 232)\
	macro(crand, 19, 21, 257)\
	macro(crandc, 19, 21, 129)\
	macro(crnand, 19, 21, 225)\
	macro(crorc, 19, 21, 417)\
	\
	macro(_wc_debug, IPO_NOARG, 21, ISO_WC_DEGUB)\
	macro(_oovpa, IPO_OOVPA, 0, 0)\
	macro(_gcm_dol_loaded, IPO_NOARG, 21, ISO_GCM_DOL_LOADED)\
	macro(_osreport, IPO_NOARG, 21, ISO_OSREPORT)\
	macro(mcrfs, 63, 21, 64)\
	macro(_dsp_quick_execute, IPO_NOARG, 21, ISO_DSP_QUICK_EXECUTE)\
	//macro(addic_, 13, 0, 0)
//macro(stwcx_, 31, 21, 150)
#define UNIMPLEMENTED_OPCODES(macro) \
	macro(eciwx, 31, 21, 310)\
	macro(ecowx, 31, 21, 438)\
	macro(fnabs, 63, 21, 136)\
	macro(lswx, 31, 21, 533)\
	macro(mtfsfi, 63, 21, 134)\
	macro(stfsux, 31, 21, 695)\
	macro(sthux, 31, 21, 439)\
	macro(tw, 31, 21, 4)\

#define DECLARE_OPFUNC(opasm, pri, sm, sec) \
	void _##opasm();

#define DECLARE_DISASMFUNC(opasm, pri, sm, sec) \
	void disasm_##opasm(Container<char>& buffer);

OPCODES(DECLARE_DISASMFUNC);
void disasm_addic_(Container<char>& buffer);
void disasm_stwcx_(Container<char>& buffer);

namespace disassembly {
	extern DWORD opcode, cia;
}

class CycleCounter : public CycleCounterBase {
public:
	CycleCounter();
	void addCycles(DWORD c);
	void endPeriod();
	const QWORD &cycles_current_period;
};

struct OVP {
	DWORD opcode, offset;
};

#define CACHE_BLOCK_SIZE 32 //bytes
#define CACHE_BLOCK_MASK ~0x1F

#define RUNSTOP (/*g::dual_run ||*/ g::disassembly_mode)
#define BLOCK_SIZE_EXPONENT 14

extern string g_errorstring; //Filled when run returns false (and GLE==0)?

class CoreBase {  //Base class for emulator core
public:
	CoreBase(MemInterface &mem, CORE_ARGUMENTS a);
	virtual ~CoreBase();
	bool load(const char *filename);
	virtual bool disassemble() = 0;
	Hardware &getHardware() { return h; }
	virtual string getdasma(DWORD address) = 0;
	virtual string getdasmo(DWORD cia, DWORD opcode) = 0;
	QWORD getCycles() const { return cc.cycles; }
	virtual DWORD getNIA() = 0;
	const Registers &getRegisters() const { return r; }
	bool run(DWORD runstop, bool degub, QWORD degub_skip, bool dumpmem);

	void setQuitSignal(const char *source) {
		p.q = true;
		h.setQuitSignal();
		DEGUB("Quit signal set by %s.\n", source);
	}
	bool quit() { return p.q; }
	void dumpmem() const;
	void getTimings(std::vector<TIMING> &timings);
	const char *getGameName();
	const char *getGameCode();

	//should be called once per second if emu is running && g::verbose
	void getVerboseText(ostream& str);
protected:
	virtual bool handle_load() = 0;
	virtual bool _run(DWORD runstop, bool degub, QWORD degub_skip, bool dumpmem) = 0;
	bool read_dol(const char *dolname);
	bool dol_bios_emu(DWORD ArenaLo);
	void setQuit() { p.q = true; DEGUB("Quit signal set\n"); }

	struct SECTION {
		DWORD address, size;
	};
	vector<SECTION> text_sections, data_sections;
	DWORD bss_address, bss_size, entry_point;

	struct SYMBOL {
		DWORD paddress, size;
		string name;
	};
#define INVALID_ADDRESS -1
	vector<SYMBOL> symbol_array;
	Array<size_t> symbol_pmap; //contains indices into symbol_array
	void do_symbol(DWORD address);
	size_t get_symbol_index(DWORD address);
	void degub_symbol(size_t index, DWORD address);
	//bool mapping_on;

	typedef void (*asm_popf)(Container<char>& buffer);
	OpArray<asm_popf> asmarr;
	CycleCounter cc;
	Registers r;
	std::list<PENDING_INTERRUPT> pending_interrupts;
	friend class InterruptRaiser;
	InterruptRaiser interrupt;
	DWORD periods;
	Hardware h;
	bool last_instruction_was_branch;

	DWORD getTBL();
	DWORD getTBU();
	bool do_interrupts();
	virtual bool take_interrupt(DWORD requested_nia) = 0;

	//lwarx / stwcx. RESERVE
	bool reserve_bit;
	DWORD reserve_data, reserve_address;	//yeah, this should actually work :)
	//except in one circumstance:
	//overwrite with the same data: my stwcx should fail, but won't.

	//spr stuff. DEC and BAT support for now
	bool mfspr_requires_processing(DWORD spr) const;
	bool mtspr_requires_processing(DWORD spr) const;
	void mfspr_with_processing(DWORD opcode);
	void mtspr_with_processing(DWORD opcode);

	//Paired Single
#define PS0(fr) (r.fpr[fr].d)
#define PS1(fr) (r.ps1[fr])

	struct GQR_PROCESSED {
		float ld_scale, st_scale;
		BYTE ld_type, st_type;
	} gqrp[8];
	void quantized_load_W0(DWORD frD, DWORD address, DWORD I);
	void quantized_load_W1(DWORD frD, DWORD address, DWORD I);
	void quantized_store_W0(DWORD address, DWORD I, DWORD frS);
	void quantized_store_W1(DWORD address, DWORD I, DWORD frS);

	void wc_debug();
	void osreport();
	void dsp_quick_execute();

#include "hle.h"
	void do_oovpa();
private:
	//loading stuff
	bool load_dol(const char *filename);
	bool load_elf(const char *filename);
	bool load_gcm(const char *filename);
	bool load_map(const char *filename);

	//other stuff
	void set_interrupt_registers(WORD vector, DWORD srr0);
	MemInterface &m;
	struct ZEROABLE_PRIVATE {
		bool q, running;
		HWND hWnd;
		ULI rtc_start, rtc_completed;
		ULI dec_wt, tbl_wt, tbu_wt, event_time;
#define GCMNAME_SIZE (0x400-0x20)
		char gamename[GCMNAME_SIZE], gamecode[5];
		DWORD tbl_reads;
	} p;

	list<TIMED_EVENT> timed_events; //sorted by time, with the earliest in front

	friend void dec_event(CoreBase *);
	void set_dec_event(ULI event_time); //must be called whenever dec is set...
	DWORD getDEC();

	inline void setCT(ULI *uli);	//Current Time. Costly, use as little as possible.
	inline ULI getCT();
};

#endif	//COREBASE_H
